"use strict";
/***********************************************************/
/*  BitSight Archer Integration - Alerts                   */
/***********************************************************/

/*
Purpose: 
	Gets BitSight alerts for companies with an appropriate subscription/settings in the BitSight Portal

High Level Overview:
	Get list of all Archer BitSight portfolio records so we have the guid for associating alert records to the right portfolio
	Use the BitSite API to retrieve the current alerts
	Create alert records in Archer's BitSight Alerts and Breaches application

Detailed Process: 
	1. Logs into Archer to get session token
	2. Retrive Customer Name from BitSight to pass into x-BitSight-Customer header param
	3. Obtain list of all Archer BitSight Portfolio records which have a guid
	4. Use BitSight API to obtain latest alerts for entire portfolio
		Iterate through if more than the defined bitsight_limit of records returned 
	5. Obtain the Archer FieldIDs for the content post via api from the app
	6. Obtain Archer values list IDs programmatically via api
	7. Builds postbody for each alert record
	8. Attempts to create alert record(s) in Archer in the BitSight Alerts and Breaches application
*/

/**********************************************************/
/* VERSIONING                                             */
/**********************************************************/
/*  
	2/2/2022 - Version 2.0
    Initial Version - 
*/



/********************************************************/
/********	ARCHER DFM APPROVED LIBRARIES
/********************************************************/
var request = require('request');
//var xpath = require('xpath');
var xmldom = require('xmldom');
var xml2js = require('xml2js');


/********************************************************/
/***** DEFAULT PARAMS
/********************************************************/
/* These params will be used unless overriden by parameters passed in. */
var defaultParams = {
	'archer_username':"dfm_BitSightAPINightly",				//Example used in Installation Guide. Update name if you change it.
	'archer_password':"YOURPASSWORD",
	'archer_instanceName':'INSTANCENAME', 					//Example: 88555 or Dev						Instance name is case-sensitive. SaaS instances are typically numbers
	'archer_webroot':"https://ARCHERURL.com/",				//Example: https://[ArcherDomainURL]/		Must include trailing slash
	'archer_webpath':"RSAarcher/", 							//Example: RSAarcher/						Must include trailing slash UNLESS there is no virtual directory...then leave blank - Preconfigured for Archer SaaS environment
	'archer_ws_root':"RSAarcher/ws/",						//Example: RSAarcher/ws/					Must include trailing slash and vdir is case-sensitive - Preconfigured for Archer SaaS environment
	'archer_rest_root':"RSAArcher/platformapi/",				//Example: RSAarcher/platformapi/			Must include trailing slash and vdir is case-sensitive - Preconfigured for Archer SaaS environment

	'bitsight_token':"YOUR_BITSIGHT_TOKEN",     				//BitSight API Token for authentication


	//Web call options
	'proxy':'',
	'verifyCerts':'true',


	//Archer Application Names
	'archer_BitSightAlertsApp':'BitSight Alerts and Breaches', 			//Name of the "BitSight Portfolio" ODA in Archer for your organization. 


	//Typically these will not change
    'bitsight_webroot':"https://api.bitsighttech.com/",     			//Main BitSight API URL
    'bitsight_alerts':"ratings/v2/alerts/latest",    					//Endpoint for latest ratings
	'bitsight_currentuser':"users/current",       						//Endpoint for getting current user/company info
	
	'bitsight_recordlimit':"1000",		//Limit of records returned from BitSight API for iteration (Default: 1000)

	//Typically these will not change unless RSA changes the structure	
	'archer_loginpath':"general.asmx",			//Example: general.asmx
    'archer_searchpath': "search.asmx",			//Example: search.asmx
	'archer_contentpath': "core/content",		//Example: core/content
	'archer_applicationpath': "core/system/application/",							//Example: core/system/questionnaire						Must include trailing slash
	'archer_fielddefinitionapppath': "core/system/fielddefinition/application/",	//Example: core/system/fielddefinition/application/			Must include trailing slash
	'archer_fielddefinitionpath': "core/system/fielddefinition/",					//Example: core/system/fielddefinition/						Must include trailing slash
	'archer_valueslistvaluepath':"core/system/valueslistvalue/flat/valueslist/",	//Example: core/system/valueslistvalue/flat/valueslist/		Must include trailing slash
	'archer_version':"core/system/applicationinfo/version",							//Example: core/system/applicationinfo/version				Do NOT include trailing slash

	//*****************SaaS/Hosted Specific Fields************************
	'bIsSaaS':"true", //Set to true for SaaS/Hosted clients
	'bDFDebugging':"false" //Only applicable for Data Feed Jobs. Always set to false for SaaS.
};

var SaaSParams =  {};

var bVerboseLogging = false; //Verbose logging will log the post body data and create output files which takes up more disk space
var bIsLegacyIntegration = false; //Set this to true if you are using the Archer 6.7 version of the package. If you are using Archer 6.10 P1 or higher, set this to: false

var appPrefix = '230'; //Used to store files in logs subdirectory when VerboseLogging is enabled.

//bSkipCompanyIfError
//If set to true, will skip over a BitSight company rating request for URL8 (logging a warning) instead of erroring/stopping the entire process.
//We recommend setting this to true to ensure the vast majority of requests work. Occassionally a single API request could timeout and probably shouldn't terminate the entire process.
var bSkipCompanyIfError = true; 


var httpTimeout = 60000; //Global setting for the http timeout per http request. Default: 60000

/********************************************************/
/********	DEBUGGING/TESTING SETTINGS
/********************************************************/
var testingMode_Tokens = {
	'LastRunTime':'2018-06-13T18:31:41Z',
	'PreviousRunContext':''
};

var testingMode_Params = {};

/********************************************************/
/***** GLOBAL VARS
/********************************************************/
var testingMode = true;
var sSessionToken = null;
var params = null;
var tokens = null;
var verifyCerts = false;
var useProxy = false;
var errorArray = [];
var previousRunContext = {};
var aArcherBitSightPortfolio = [];
var aBitSightAlerts = []; //Final object returned to Archer data feed
var ArcherValuesLists = []; //This will be used for iterating through to obtain values list value id from Archer
var bSaaSErrorFound = false;
var sDebug = '';
var iCurrentAlert = 0;
var sArcherVersionNum = "TBD";

//Just a simple way to remember the Archer input data types
var ArcherInputTypes = 
{
    'text':1,
    'numeric':2,
    'date':3,
    'valueslist':4,
    'externallinks':7,
    'usergrouprecperm':8,
    'crossref':9,
    'attachment':11,
    'image':12,
    'matrix':16,
    'ipaddress':19,
    'relatedrecords':23,
    'subform':24
}

var COST_ArcherBitSightVersion = "RSA Archer 1.0";
var COST_Platform = "RSA Archer";
var BitSightCustomerName = "unknown";

/********************************************************/
/********	INIT
/********************************************************/
function init(){
	/* run the feed */
	//LogInfo("Datafeed Init");
	/* check if testing mode should be active (no archer DFE present) */
	if(typeof context == 'undefined'){
		testingMode=true;
	}
	/* get params and tokens */
	params = getArcherParams();
	tokens = getArcherTokens();
    /* setup cert verify */
    if(params['verifyCerts'].toLowerCase()=='true'){
        verifyCerts=true;
	}else{
		process.env.NODE_TLS_REJECT_UNAUTHORIZED = "0";
	}
	/* setup proxy */
	if(params['proxy']!=null && params['proxy']!=''){
		useProxy=true;
	}
	/* get the last run time */
	var lrt = new Date('1970-01-10T00:00:00Z');
	/* check for last run time */
	if('LastRunTime' in tokens && tokens['LastRunTime']!=null && tokens['LastRunTime']!=''){
		lrt = new Date(tokens['LastRunTime']);
	}
	//LogInfo("Last Datafeed Run Time: " + lrt.toISOString());
	return lrt;
}

/********************************************************/
/********	DATA CONVERSIONS
/********************************************************/
function jsonToXMLString(json,rootElement){
	if(rootElement){
		var bldrOpts = {
			headless: true, rootName: rootElement, renderOpts: {'pretty': true,'indent': '    ','newline': '\r\n','cdata': true}
		}
	}else{
		var bldrOpts = {
			headless: true, renderOpts: {'pretty': true,'indent': '    ','newline': '\r\n','cdata': true}
		}
	}
	return new xml2js.Builder(bldrOpts).buildObject(json);
}

function jsonToString(json){
	return JSON.stringify(json,null,4);
}

function xmlStringToJSON(xml,callBack){
	xml2js.parseString(xml, {}, callBack);
}

function xmlStringToXmlDoc(xml){
	var p  = new xmldom.DOMParser;
	return p.parseFromString(xml);
}

function jsonArrayToXMLBuffer(jsonArray,elementName){
	/* holds the buffers */
	var buffers = [];
	/* conver each element to an xml buffer */
	for(var i in jsonArray){
		/* convert it to xml */
		var xmlString = jsonToXMLString(jsonArray[i],elementName);
		/* convert it to a buffer */
		var b = Buffer.from(xmlString+'\n', 'utf8');
		/* add to buffers array */
		buffers.push(b);
	}
	/* concat to one giant buffer */
	return Buffer.concat(buffers);
}

/********************************************************/
/********	ARCHER CALLBACK INTERFACE
/********************************************************/
function getArcherParams(){
	var params = null;
	/* which params should we use? */
	if(testingMode){
		params=testingMode_Params;
	}else{
		params=context.CustomParameters;
	}
	/* now add the default params */
	for(var k in defaultParams){
		if(!(k in params)){
			params[k]=defaultParams[k];
		}
	}
	/* return them */
	return params;
}

function getArcherTokens(){
	if(testingMode){
		return testingMode_Tokens;
	}
	return context.Tokens;
}

var dfCallback = function(err,data){
	if(err){
		LogError("ERROR: Datafeed Failure due to error.");
		if (params['bIsSaaS'] == "true"){
			LogSaaSError();
		}else{
			if (params['bDFDebugging'] == "true" || params['bDFDebugging'] == true)
			{
				LogInfo("Data Feed Debugging Enabled");
				//The commented line sends the data to a record in Archer, but the record ID needs to be provided.
				//var start = Buffer.from('<Records><Record><InactiveUserConfigContentID>774463</InactiveUserConfigContentID><Report_Results_HTML><![CDATA[\n', 'utf8');
				var start = Buffer.from('<Records><Record><![CDATA[\n', 'utf8');
				var middle = Buffer.from(sDebug,'utf8');
				//var end = Buffer.from(']]></Report_Results_HTML></Record></Records>\n', 'utf8');
				var end = Buffer.from(']]></Record></Records>\n', 'utf8');
				var returntoArcher = Buffer.concat([start,middle,end]);
				var sXMLReturn = returntoArcher.toString('utf8');
				sendDataToArcher(sXMLReturn);
			}
			else{
				sendDataToArcher(null);
			}
		}
	}else{
		if (params['bIsSaaS'] == "true"){
			LogSaaSSuccess();
			LogInfo("SUCCESS: Overall Success");
		}else{		
			LogInfo("Exporting Data to Archer.");
			sendDataToArcher(data);
		}
	}
};

function sendDataToArcher(data)
{
	if(testingMode || bVerboseLogging){
		if (data){
			var fs = require('fs'); 
			fs.writeFile("logs\\" + appPrefix + "\\" + "!ReturnedToArcher.xml",data,function(err)
			{
				if(err) 
				{
					LogError("ERROR SAVING FILE IN TEST MODE: " + err);
				}
			});
		}
		if(errorArray.length>0){
			for(var i in errorArray){
				LogError(errorArray[i]);
			}
		}
	}
	else{
		/* check for any errors */
		if (errorArray.length>0){
			//callback(errorArray,{output: data, previousRunContext: JSON.stringify(previousRunContext)}); //Only if data came back as JSON
			callback(errorArray,{output: data, previousRunContext: JSON.stringify(previousRunContext)});  //Attempting to use the XML returned from the Archer API
		}else{
			//callback(null,{output: data, previousRunContext: JSON.stringify(previousRunContext)});  //Only if data came back as JSON
			callback(null,{output: data, previousRunContext: JSON.stringify(previousRunContext)}); //Attempting to use the XML returned from the Archer API
		}
	}
}

/********************************************************/
/********	ERROR HANDLING AND LOGGING
/********************************************************/
function captureError(text){
	LogSaaSError();

	// if(text!=null)
	// {
	// 	/* create a new error to get the stack */
	// 	var e = new Error();
	// 	/* create error string for array */
	// 	var errString = text + "\n" + e.stack;
	// 	/* add to error array */
	// 	errorArray.push(errString);
	// }

}

function getDateTime(){
    var dt = new Date();
    return pad(dt.getFullYear(),4)+"-"+pad(dt.getMonth()+1,2)+"-"+pad(dt.getDate(),2)+" "+pad(dt.getHours(),2)+":"+pad(dt.getMinutes(),2)+":"+pad(dt.getSeconds(),2);
}

function LogInfo(text){
    console.log(getDateTime() + " :: INFO  :: " + text);
	if (params['bDFDebugging'] == "true" || params['bDFDebugging'] == true)
	{
		appendDebug(text);
	}
}

function LogError(text){
	LogSaaSError();
	console.log(getDateTime() + " :: ERROR :: " + text);
	if (params['bDFDebugging'] == "true" || params['bDFDebugging'] == true)
	{
		appendDebug(text);
	}
}

function LogWarn(text){
    console.log(getDateTime() + " :: WARN  :: " + text);
	if (params['bDFDebugging'] == "true" || params['bDFDebugging'] == true)
	{
		appendDebug(text);
	}
}

function LogVerbose(text){
	if (bVerboseLogging){
		LogInfo(text);
	}
}

function pad(num, size){
    var s = num+"";
    while (s.length < size) s = "0" + s;
    return s;
}

/********************************************************/
/********	LogSaaSError
/********************************************************/
var LogSaaSError = function()
{
	//Simple method to log an error to a file for SaaS for batch execution. The batch file will check if this file exists.
	if (params['bIsSaaS'] == "true" && bSaaSErrorFound == false){
		bSaaSErrorFound = true;  //Only log if we haven't found an error to help avoid file lock issues.
		var fs = require('fs'); 			
		fs.writeFileSync("logs\\error-" + appPrefix + ".txt","ERROR");
		LogInfo("Logged error and created logs\\error-" + appPrefix + ".txt file.");
	}
}

/************************************************************************************************************************************************************************************************************************************/
/********	LogSaaSSuccess
/************************************************************************************************************************************************************************************************************************************/
var LogSaaSSuccess = function()
{
	//Simple method to log successful execution for SaaS for batch execution. The batch file will check if this file exists.
	if (params['bIsSaaS'] == "true" && bSaaSErrorFound == false){
		var fs = require('fs'); 			
		fs.writeFileSync("logs\\success-" + appPrefix + ".txt","SUCCESS");
		LogInfo("Logged success and created logs\\success-" + appPrefix + ".txt file.");
	}
}



//GetArcherText validates then returns data from a text value from Archer
function GetArcherText(sText)
{
	if (typeof sText == 'undefined' || 
		typeof sText._ == 'undefined' || 
		sText == null)
	{
		return "";
	}else{
		return sText._;
	}
}

//GetBitSightText validates then returns data from a text value from BitSight
function GetBitSightText(sText)
{
	if (typeof sText == 'undefined' ||
		sText == null)
	{
		return "";
	}else{
		return sText;
	}
}


//GetArcherValue validates then returns data from a single value list
function GetArcherValue(jValueNode)
{
	if (typeof jValueNode == 'undefined' || 
		jValueNode == null ||
		typeof jValueNode.ListValues == 'undefined' || 
		typeof jValueNode.ListValues[0] == 'undefined' || 
		typeof jValueNode.ListValues[0].ListValue == 'undefined' || 
		typeof jValueNode.ListValues[0].ListValue[0] == 'undefined' || 
		typeof jValueNode.ListValues[0].ListValue[0]._ == 'undefined')
	{
		return "";
	}else{
		return jValueNode.ListValues[0].ListValue[0]._;
	}
}


function GetSeverityValuesListLookup(sInput)
{
    if (typeof sInput == 'undefined' || sInput == null || sInput == "" || sInput == "N/A")
    {
        return "";
    }
    sInput = sInput.toLowerCase();
    if (sInput == 'critical')
    {
        return SaaSParams['severity_Critical'];
    }else if (sInput == 'increase')
    {
        return SaaSParams['severity_Increase'];
    }else if (sInput == 'informational')
    {
        return SaaSParams['severity_Informational'];
    }else if (sInput == 'warn')
    {
        return SaaSParams['severity_Warn'];
    }
}

/********************************************************/
/********	WEB CALL
/********************************************************/
function webCall(url,method,headers,postBody,callBack)
{
	/* build options */
	var options = {
		method: method,
		uri: url,
		headers: headers,
		body: postBody,
		rejectUnauthorized: verifyCerts
	}
	/* add in proxy */
	if(useProxy)
	{
		options['proxy']=params['proxy'];
	}
	
	//LogInfo("HERE url: " + url);
	//LogInfo("HERE method: " + method);
	//LogInfo("HERE postBody: " + postBody);
	
	/* make the request */
	request(options,
		function handleResponse(err, response, body)
		{
			var errMessage = null;
			var headers = {};				
			/* check for error */
			if(err)
			{
				errMessage='HTTP ERROR: ' + err;
			}
			else
			{
				headers = response.headers;
				//LogInfo("HERE headers: " + headers);
				if(response.statusCode!=200)
				{
					errMessage='HTTP ' + response.statusCode + ' ERROR: ' + err;
				}
			}
			/* check for error */
			if(errMessage)
			{
				captureError('Error in HTTP Response: ' + errMessage);
				callBack(true,headers,body);
			}
			else
			{
				callBack(false,headers,body);
			}
		}
	);
}

function appendDebug(sTxt)
{
	sDebug+=sTxt.toString('utf8') + "\n";
	//sDebug+=sTxt + "|";
}

function b64Encode(str) {  
    var CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";  
    var out = "", i = 0, len = str.length, c1, c2, c3;  

     while (i < len) {  
                      c1 = str.charCodeAt(i++) & 0xff;  
                      if (i == len) {  
                                       out += CHARS.charAt(c1 >> 2);  
                                       out += CHARS.charAt((c1 & 0x3) << 4);  
                                       out += "==";  
                                       break;  
     }  

     c2 = str.charCodeAt(i++);  
     if (i == len) {  
                      out += CHARS.charAt(c1 >> 2);  
                      out += CHARS.charAt(((c1 & 0x3)<< 4) | ((c2 & 0xF0) >> 4));  
                      out += CHARS.charAt((c2 & 0xF) << 2);  
                      out += "=";  
                      break;  
     }  

     c3 = str.charCodeAt(i++);  
     out += CHARS.charAt(c1 >> 2);  
     out += CHARS.charAt(((c1 & 0x3) << 4) | ((c2 & 0xF0) >>4));  
     out += CHARS.charAt(((c2 & 0xF) << 2) | ((c3 & 0xC0) >>6));  
     out += CHARS.charAt(c3 & 0x3F);  
    }  
     return out;  
} 

function makeBasicAuth(token)
{
    //Purpose of this is to convert the token to the authorization header for basic auth
    //Format is Token with a colon at the end then converted to Base64
    return b64Encode(token + ":");
}

/********************************************************/
/********	ArcherAPI_Login
/********************************************************/
function ArcherAPI_Login(callBack)
{
	appendDebug("Login");
	/* build url */
	var url =  params['archer_webroot'] + params['archer_ws_root'] + params['archer_loginpath'];

	var postBody = "<?xml version=\"1.0\" encoding=\"utf-8\"?><soap:Envelope xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:soap=\"http://schemas.xmlsoap.org/soap/envelope/\">" +
				   "<soap:Body><CreateUserSessionFromInstance xmlns=\"http://archer-tech.com/webservices/\"><userName>" + params['archer_username'] + "</userName><instanceName>" + params['archer_instanceName'] + "</instanceName><password>" + params['archer_password'] + "</password></CreateUserSessionFromInstance></soap:Body></soap:Envelope>";
	
	var headers = {
			'content-type' : 'text/xml; charset=utf-8'
	}
	
	var results = []; 

	var loginToArcher = function(url){
		LogInfo('----------------------------Login Attempt Starting------------------');
		LogInfo('URL1: ' + url);
		//LogVerbose('POST1: ' + postBody);

		//LogInfo('HEAD: ' + headers);

		webCall(url,'POST',headers,postBody,function(err,h,b)
		{
			if(err)
			{
				LogError("ERR1: " + err);
				//LogInfo("HERE HEAD1: " + h);
				LogError("RESP1: " + b);
				appendDebug("ERR1");
				callBack(true,null);
			}
			else
			{
				//LogInfo("webCall Results: " + b);
				results = b; //This is the result from the webCall
				var doc = xmlStringToXmlDoc(results);
				//Get the value of the the text which is the session token
				sSessionToken = doc.getElementsByTagName("CreateUserSessionFromInstanceResult")[0].childNodes[0].nodeValue;
				LogInfo("sSessionToken: " + sSessionToken);
				if (sSessionToken.length > 5)
				{
					GetArcherVersion(callBack);
				}
				else
				{
					LogError("sSessionToken blank: " + sSessionToken);
					appendDebug("SessionBlank");
					callBack(true,null);
				}
			}
		});
	}

	/* kick off process */
	loginToArcher(url);
}

/********************************************************/
/********	GetArcherVersion
/********************************************************/
var GetArcherVersion = function(callBack)
{
	LogInfo('----------------------------GetArcherVersion----------------------------');

	var url =  params['archer_webroot'] + params['archer_rest_root'] + params['archer_version'];

	var headers = {	'Authorization' : 'Archer session-id="' + sSessionToken + '"',
					'Content-type': 'application/json',
					'Accept': 'application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8'};
	
	var postBody = {};

	LogInfo('URL1.1: ' + url);
	LogVerbose('BODY1.1: ' + jsonToString(postBody));
	//LogInfo('HEAD: ' + headers);

	webCall(url,'GET',headers,jsonToString(postBody),function(err,h,b)
	{
		if(err)
		{
			LogSaaSError();
			LogError("ERR1.1: " + err);
			//LogInfo("HERE HEAD1: " + h);
			LogError("BODY1.1: " + b);
		}
		else
		{
			LogVerbose("Raw Report Results: " + b);
			var resultJSON = JSON.parse(b);

			if(typeof resultJSON != 'undefined' && typeof resultJSON.RequestedObject != 'undefined' && resultJSON.RequestedObject.Version != 'undefined')
			{
				sArcherVersionNum = resultJSON.RequestedObject.Version; //Get the Version
				COST_Platform = COST_Platform + " (" + sArcherVersionNum + ")";
				LogInfo("sArcherVersionNum: " + sArcherVersionNum);
				GetCurrentBitSightCompanyInfo(callBack);
			}
			else
			{
				sArcherVersionNum = "Unknown";
				LogWarn("ERROR Obtaining Archer Version Number: " + b);
				GetCurrentBitSightCompanyInfo(callBack);
			}
		}
	});
}

/************************************************************************************************************************************************************************************************************************************/
/********	GetCurrentBitSightCompanyInfo
/************************************************************************************************************************************************************************************************************************************/
function GetCurrentBitSightCompanyInfo(callBack)
{
	var aBitSightCompanyInfo;
	var bitSightAPI_GetCurrentBitSightCompanyInfo = function(callBack)
	{
        var url = params['bitsight_webroot'] + params['bitsight_currentuser'];

        var sAuthorization = makeBasicAuth(params['bitsight_token']);

        var headers = { 'Accept' : 'application/json',
                        'Authorization' : 'Basic ' + sAuthorization
					};

        var postBody = null;
		
		LogInfo("----------------------GetCurrentBitSightCompanyInfo----------------------");
		LogInfo('URL2: ' + url);
		LogInfo('HEAD2: ' + JSON.stringify(headers));

		/* build options */
		var options = {
			method: 'GET',
			uri: url,
			headers: headers,
			body: postBody,
			timeout: httpTimeout,
			rejectUnauthorized: verifyCerts
		}
		
		/* make the request */
		request(options,function (err1, response1, body)
		{
			var errMessage = null;
			
			/* check for error */
			if(err1)
			{

				errMessage='HTTP ERROR: ' + err1;
			}
			else
			{
				//LogInfo("Header Response: " + response1.headers);
				if(response1.statusCode!=200)
				{
					errMessage='HTTP ' + response1.statusCode + ' ERROR: ' + err1;
				}
			}
			/* check for error */
			if(errMessage)
			{
				LogError("ERR2: " + errMessage);
				LogError("BODY2: " + body);
				captureError('Error in HTTP Response: ' + errMessage);
				callBack(true,null);
			}
			else
			{
                LogInfo("body:" + body);
				aBitSightCompanyInfo = JSON.parse(body);
                LogInfo("----------------------GOT Company Info----------------------");
                
                //Just for testing...save to file...
                if (bVerboseLogging === true)
                {
                    var fs = require('fs'); 
                    fs.writeFileSync("logs\\" + appPrefix + "\\" + "aBitSightCompanyInfo.json", JSON.stringify(aBitSightCompanyInfo));
                    LogInfo("Saved to logs\\" + appPrefix + "\\" + "aBitSightCompanyInfo.json file");
                }

				if (typeof aBitSightCompanyInfo == 'undefined' || 
					typeof aBitSightCompanyInfo.customer == 'undefined' || 
					typeof aBitSightCompanyInfo.customer.name == 'undefined')
				{
					BitSightCustomerName = "unknown";
				}else{					
					BitSightCustomerName = aBitSightCompanyInfo.customer.name;
					LogInfo("BitSight Customer: " + BitSightCustomerName);
				}

				GetArcherBitSightCompanies(callBack);
			}
		});
	}

	/* kick off process */
    bitSightAPI_GetCurrentBitSightCompanyInfo(callBack);
}

/************************************************************************************************************************************************************************************************************************************/
/********	GetArcherBitSightCompanies
/************************************************************************************************************************************************************************************************************************************/
function GetArcherBitSightCompanies(callBack)
{
	/* build url */
	var url =  params['archer_webroot'] + params['archer_ws_root'] + params['archer_searchpath'];

	var headers = {'content-type' : 'text/xml; charset=utf-8',
			'SOAPAction' : 'http://archer-tech.com/webservices/ExecuteSearch'}

	//Building a search criteria instead of using the report. This way we can return thousands of records instead of having to page through 250 records at a time...
	var sSearchCriteria;
	if (!bIsLegacyIntegration)
	{	
		sSearchCriteria =   '<SearchReport><PageSize>20000</PageSize><MaxRecordCount>20000</MaxRecordCount><DisplayFields>' +
							'<DisplayField name="BitSight Company Name">d616bf90-7cb7-4858-8f26-fddf66d618c8</DisplayField>' +
							'<DisplayField name="BitSight GUID">ffb06a0b-a5bd-4fff-9250-76f6a757d926</DisplayField>' +
							'</DisplayFields><Criteria><ModuleCriteria><Module name="BitSight Portfolio">21d4cfb5-e4d2-4c0b-a34f-559e3a0ef004</Module>' +
							'<SortFields><SortField name="Sort1"><Field name="BitSight Company Name">d616bf90-7cb7-4858-8f26-fddf66d618c8</Field>' +
							'<SortType>Ascending</SortType></SortField></SortFields></ModuleCriteria><Filter><Conditions><TextFilterCondition name="Text 1">' +
							'<Field name="BitSight GUID">ffb06a0b-a5bd-4fff-9250-76f6a757d926</Field><Operator>DoesNotEqual</Operator><Value>' +
							'</Value></TextFilterCondition></Conditions></Filter></Criteria></SearchReport>';
	}
	else
	{
		sSearchCriteria =   '<SearchReport><PageSize>20000</PageSize><MaxRecordCount>20000</MaxRecordCount><DisplayFields>' +
							'<DisplayField name="BitSight Company Name">C9ABC083-CC49-498D-8ACA-FD1BEE4C9A5A</DisplayField>' +
							'<DisplayField name="BitSight GUID">6E965C7C-84A7-4A98-9701-2EF0E980D967</DisplayField>' +
							'</DisplayFields><Criteria><ModuleCriteria><Module name="BitSight Portfolio">54FE69B0-3C28-4D94-852D-841151634F22</Module>' +
							'<SortFields><SortField name="Sort1"><Field name="BitSight Company Name">C9ABC083-CC49-498D-8ACA-FD1BEE4C9A5A</Field>' +
							'<SortType>Ascending</SortType></SortField></SortFields></ModuleCriteria><Filter><Conditions><TextFilterCondition name="Text 1">' +
							'<Field name="BitSight GUID">6E965C7C-84A7-4A98-9701-2EF0E980D967</Field><Operator>DoesNotEqual</Operator><Value>' +
							'</Value></TextFilterCondition></Conditions></Filter></Criteria></SearchReport>';
	}
	
	LogVerbose('sSearchCriteria: ' + sSearchCriteria);

	//Must escape the XML to next inside of the soap request... 
	sSearchCriteria = sSearchCriteria.replace(/&/g, '&amp;')
	               .replace(/</g, '&lt;')
	               .replace(/>/g, '&gt;')
	               .replace(/"/g, '&quot;')
	               .replace(/'/g, '&apos;');
	
	var postBody = "<?xml version=\"1.0\" encoding=\"utf-8\"?><soap:Envelope xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:soap=\"http://schemas.xmlsoap.org/soap/envelope/\">" +
				"<soap:Body><ExecuteSearch xmlns=\"http://archer-tech.com/webservices/\"><sessionToken>" + sSessionToken + "</sessionToken>" +
				"<searchOptions>" + sSearchCriteria + "</searchOptions><pageNumber>1</pageNumber></ExecuteSearch></soap:Body></soap:Envelope>";
		
	var results = []; 

	var archerAPI_GetArcherBitSightCompanies = function(url){
		LogInfo('----------------------------GET Archer BitSight Companies----------------------------');
		LogInfo('URL3: ' + url);
		//LogVerbose('After escaping BODY3: ' + postBody);
		//LogVerbose('HEAD3: ' + headers);

		webCall(url,'POST',headers,postBody,function(err,h,b)
		{
			if(err)
			{
				LogError("ERR3: " + err);
				//LogError("HEAD3: " + h);
				LogError("BODY3: " + b);
				callBack(true,null);
			}
			else
			{
				//LogInfo("Raw Report Results: " + b);
				results = b; //This is the result from the webCall
				reviewResults();
			}
		});
	}

	var reviewResults = function()
	{
		//Convert XML to an XMLDOM
		var doc = xmlStringToXmlDoc(results);

		//Check to see if nothing was returned from the search query
		if (typeof doc.getElementsByTagName("ExecuteSearchResult") == "undefined" || 
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0] == "undefined" || 
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0] == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0].nodeValue == "undefined")
		{
			//There weren't any records
			LogInfo("Either no results or Archer search error.");
			var xmlReturn = Buffer.from('<Records></Records>', 'utf8');
			callBack(false,xmlReturn);
			return;
		}
		else  //Need to proceed and check the count anyway.
		{
			//Need to get the xml inside the SOAP request and url decode the results
			var sXML = doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0].nodeValue;

			//variable for our json object
			var resultJSON = null;
			//turn the xml results into the json object
			xml2js.parseString(sXML, function (err, result) 
			{
				resultJSON = result; //get the result into the object we can use below;
			});
			
			if (bVerboseLogging === true)
			{
				var fs = require('fs'); 			
				fs.writeFileSync("logs\\" + appPrefix + "\\" + "archerBitSightCompanies.json", JSON.stringify(resultJSON));
				LogInfo("Saved to logs\\" + appPrefix + "\\" + "archerBitSightCompanies.json file");
			}
			
			var iNumCompanies = resultJSON.Records.$.count; //Get the number of record returned
			LogInfo("iNumCompanies=" + iNumCompanies);

			//Check to see if we have any existing records
			if (iNumCompanies == 0)
			{
				LogInfo("No existing Archer BitSight records. Exiting.");
				callBack(false,null);
			}
			else
			{
				compileResults(resultJSON)
			}
		}
	}

	//Results were found, now get the details
	var compileResults = function(resultJSON)
	{
		var iID_BitSightCompanyGUID;
		var iID_BitSightCompanyName;

		//Get the LevelID of the Portfolio app
		var sArcherBitSightPortfolio_LevelID = resultJSON.Records.LevelCounts[0].LevelCount[0].$.id;

		//Iterate through the FieldDefinition to get the field ids that we care about
		for (var h in resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition) 
		{
			var sAlias = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.alias;
			var sID = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.id;
				 		
			//LogInfo("  Alias:" + sAlias + "   ID: " + sID);
			if (sAlias == "BitSight_Company_Name"){iID_BitSightCompanyName = sID;}
			else if (sAlias == "BitSight_GUID"){iID_BitSightCompanyGUID = sID;}
		}
		
		var sContentID = "";
		var sBitSightCompanyName;
		var sBitSightCompanyGUID;

        //Iterate through each Archer company record....
		for (var i in resultJSON.Records.Record)
		{
			LogInfo("-----ARCHER COMPANY RECORD #" + i + "-------");
			sContentID = null;
			sBitSightCompanyName= "";
			sBitSightCompanyGUID= "";
			
			//Get the Tracking ID...
			sContentID = resultJSON.Records.Record[i].$.contentId;
			//LogInfo("sContentID: " + sContentID);

			//Iterate through the Field elements for the current config record to get the goodies
			for (var y in resultJSON.Records.Record[i].Field) 
			{
				//Get the id of the field because we need to match on the ones we care about...
				sID = resultJSON.Records.Record[i].Field[y].$.id;
				
				//Now find all the good data we care about...
				if (sID == iID_BitSightCompanyName)
				{
					sBitSightCompanyName =  GetArcherText(resultJSON.Records.Record[i].Field[y]);
				}
				else if (sID == iID_BitSightCompanyGUID) 
				{
					sBitSightCompanyGUID = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				}
			}
			
			LogInfo("Content ID: " + sContentID + " CompanyName:" + sBitSightCompanyName);
			//Populate the main record with the details we care about.... 
			aArcherBitSightPortfolio[aArcherBitSightPortfolio.length] = {
				"ContentID":sContentID,
				"CompanyName":sBitSightCompanyName,
				"GUID":sBitSightCompanyGUID,
			};

			LogInfo("*Company Number#" + aArcherBitSightPortfolio.length + "=" + JSON.stringify(aArcherBitSightPortfolio[aArcherBitSightPortfolio.length-1]));
			
		}
       
		//Just for testing...save to file...
		if (bVerboseLogging === true)
		{
			var fs = require('fs'); 
            fs.writeFileSync("logs\\" + appPrefix + "\\" + "aArcherBitSightPortfolio.json", JSON.stringify(aArcherBitSightPortfolio));
			LogInfo("Saved to logs\\" + appPrefix + "\\" + "aArcherBitSightPortfolio.json file");
		}

		//Now that we have the current info, now go get the info from the BitSight API
		GetAlerts(0,callBack,null);
	}

	/* kick off process */
	archerAPI_GetArcherBitSightCompanies(url);
}


/************************************************************************************************************************************************************************************************************************************/
/********	GetAlerts
/************************************************************************************************************************************************************************************************************************************/
function GetAlerts(iOffset,callBack,nextURL)
{
	LogInfo('----------------------------GetAlerts----------------------------');
	//Example URL: https://api.bitsighttech.com/ratings/v2/alerts/latest?limit=100&offset=0&expand=details
	var url;
	if (nextURL == null){
		LogInfo("STARTING ON Offset #: " + iOffset); //Just in case we need to iterate through more results
		var sParameters = 'limit=' + params['bitsight_recordlimit'];
			sParameters+= '&offset=' + iOffset;
			sParameters+= '&expand=details';
		url = params['bitsight_webroot'] + params['bitsight_alerts'] + '?' + sParameters;
	}else{
		LogInfo("next url: " + nextURL); //Just in case we need to iterate through more results
		url = nextURL;
	}
    var sAuthorization = makeBasicAuth(params['bitsight_token']);
	var postBody = ''; 
    var headers = { 'Accept' : 'application/json',
                    'Authorization' : 'Basic ' + sAuthorization,
					'X-BITSIGHT-CONNECTOR-NAME-VERSION':COST_ArcherBitSightVersion,
					'X-BITSIGHT-CALLING-PLATFORM-VERSION':COST_Platform,
					'X-BITSIGHT-CUSTOMER':BitSightCustomerName
                    };
	
	LogInfo('URL4: ' + url);
	//LogVerbose('BODY: ' + JSON.stringify(postBody));

	/* build options */
	var options = {
		method: 'GET',
		uri: url,
		headers: headers,
		body: postBody,
		timeout: httpTimeout,
		rejectUnauthorized: verifyCerts
	}
	
	var alertsResponse;

	/* make the request */
	request(options,function (err1, response, body)
	{
		var errMessage = null;
		
		/* check for error */
		if(err1)
		{
			errMessage='HTTP ERROR: ' + err1;
		}
		else
		{
			//LogInfo("Header Response: " + response.headers);
			if(response.statusCode!=200)
			{
				errMessage='HTTP ' + response.statusCode + ' ERROR: ' + err1;
			}
		}
		/* check for error */
		if(errMessage)
		{

			//If skipping when an error is true, then skip over it and log the warning. Otherwise, terminate the entire process.
			if (bSkipCompanyIfError){
				LogWarn("bSkipCompanyIfError is set to true.");
				LogWarn("ERR4: " + errMessage);
				LogWarn("BODY4: " + body);

				//Iterate through if there were multiple pages
				if(nextURL == null)
				{
					LogInfo("nextURL was null so continuing.");
					//Just for testing...save to file...
					if (bVerboseLogging === true)
					{
						var fs = require('fs'); 			
						fs.writeFileSync("logs\\" + appPrefix + "\\" + "aBitSightAlerts.json", JSON.stringify(aBitSightAlerts));
						LogInfo("Saved to logs\\" + appPrefix + "\\" + "aBitSightAlerts.json file");
					}

					DetermineExportPath(callBack);
				}
				else
				{
					//Still have more pages to iterate through...
					LogInfo("Iterating through next url:" + nextURL);				
					GetAlerts(null,callBack,nextURL);
				}
			}
			else
			{
				LogError("ERR4: " + errMessage);
				LogError("BODY4: " + body);
				captureError('Error in HTTP Response: ' + errMessage);
				callBack(true,null);
			}
		}
		else
		{
			LogInfo("----------------------GOT GetAlerts----------------------");
			LogInfo("Raw Results body: " + body);
			alertsResponse = JSON.parse(body);

			if (typeof alertsResponse == "undefined" || 
				typeof alertsResponse.links == "undefined" ||
				typeof alertsResponse.count == "undefined" ||
				typeof alertsResponse.results == "undefined")
			{
				//Nothing returned?
				LogInfo("No alert data returned from BitSight API. Exiting.");
				return;	
			}
			else
			{
				//Get the next iteration if available. 
                if (typeof alertsResponse.links.next == "undefined")
                    {nextURL=null;}
                else{nextURL=alertsResponse.links.next;}

				//For each alert in the results, build out the array of details for import into Archer
				for (var i in alertsResponse.results)
				{

					var alertcompanyGuid = GetBitSightText(alertsResponse.results[i].company_guid);
					var bFoundCompany = false;
					var sArcherContentID;
					var sArcherCompanyName;
				
					//Now see which companies we SHOULD be monitoring in Archer
					for (var j in aArcherBitSightPortfolio)
					{
						if (aArcherBitSightPortfolio[j].GUID == alertcompanyGuid)
						{
							bFoundCompany = true;
							sArcherContentID = aArcherBitSightPortfolio[j].ContentID;
							sArcherCompanyName = aArcherBitSightPortfolio[j].CompanyName;
						}
					}

					//If we found the company in Archer, then we can add it to the alerts
					if (bFoundCompany)
					{
						//Need to convert the Alert details from JSON to nice text for humans based on the type
						var niceText = ConvertAlertDetails(	GetBitSightText(alertsResponse.results[i].alert_type),
															alertsResponse.results[i].details);

						var rating_change = GetRatingChange(GetBitSightText(alertsResponse.results[i].alert_type),
															alertsResponse.results[i].details);

						aBitSightAlerts[aBitSightAlerts.length] = {
							"ArcherContentID":sArcherContentID,
							"ArcherCompanyname":sArcherCompanyName,
							"guid":GetBitSightText(alertsResponse.results[i].guid),
							"alert_type":GetBitSightText(alertsResponse.results[i].alert_type),
							"alert_date":GetBitSightText(alertsResponse.results[i].alert_date),
							"company_guid":GetBitSightText(alertsResponse.results[i].company_guid),
							"folder_name":GetBitSightText(alertsResponse.results[i].folder_name),
							"severity":GetBitSightText(alertsResponse.results[i].severity),
							"trigger":GetBitSightText(alertsResponse.results[i].trigger),
							"details":niceText,
							"rating_change":rating_change,
							"raw":GetBitSightText(JSON.stringify(alertsResponse.results[i].details))
						}	
					}				
				}
			}

			//Iterate through if there were multiple pages
			if(nextURL == null)
			{
				LogInfo("nextURL was null so continuing.");
				//Just for testing...save to file...
				if (bVerboseLogging === true)
				{
					var fs = require('fs'); 			
					fs.writeFileSync("logs\\" + appPrefix + "\\" + "aBitSightAlerts.json", JSON.stringify(aBitSightAlerts));
					LogInfo("Saved to logs\\" + appPrefix + "\\" + "aBitSightAlerts.json file");
				}

    			DetermineExportPath(callBack);
			}
			else
			{
				//Still have more pages to iterate through...
				LogInfo("Iterating through next url:" + nextURL);				
				GetAlerts(null,callBack,nextURL);
			}
		}
	});
}


function GetRatingChange(sAlertType,jRawDetails)
{
	if (typeof sAlertType == 'undefined'){
		return "NA";
	}

	sAlertType = sAlertType.toLowerCase().trim();
	if (sAlertType == "percent_change") //PERCENT_CHANGE = {"rating_change_pct":-3,"start_rating":530,"end_rating":510}
	{
		var x = parseInt(jRawDetails.end_rating) - parseInt(jRawDetails.start_rating);
		if (!isNaN(x))
		{
			return x;
		}
		else
		{
			return "NA";		
		}
	}
	return "NA";
}

function ConvertAlertDetails(sAlertType,jRawDetails)
{
	if (typeof sAlertType == 'undefined'){
		return "No Alert Type";
	}

	sAlertType = sAlertType.toLowerCase().trim();
	if (sAlertType == "vulnerability")
	{
		return jRawDetails.message;
	}
	else if (sAlertType == "risk_category")
	{
		var direction = GetDirectionLetter(jRawDetails.start_grade,jRawDetails.end_grade);
		return jRawDetails.risk_vector + ": " + jRawDetails.start_grade + " " + direction + " " + jRawDetails.end_grade;
	}
	else if (sAlertType == "nist_category")
	{
		var direction = GetDirectionLetter(jRawDetails.start_grade,jRawDetails.end_grade);
		return jRawDetails.nist_category_display_name + ": " + jRawDetails.start_grade + " " + direction + " " + jRawDetails.end_grade;
	}
	else if (sAlertType == "percent_change") //PERCENT_CHANGE = {"rating_change_pct":-3,"start_rating":530,"end_rating":510}
	{
		var direction = GetDirectionNumber(jRawDetails.start_rating,jRawDetails.end_rating);
		return direction + " " + jRawDetails.rating_change_pct + "% (" + jRawDetails.start_rating + " " + direction + " " + jRawDetails.end_rating + ")";
	}
	else if (sAlertType == "rating_threshold") //RATING_THRESHOLD = {"rating_threshold":570,"start_rating":580,"end_rating":560}
	{
		var direction = GetDirectionNumber(jRawDetails.start_rating,jRawDetails.end_rating);
		return direction + " " + jRawDetails.rating_threshold + " (" + jRawDetails.start_rating + " " + direction + " " + jRawDetails.end_rating + ")";
	}
	else if (sAlertType == "informational" || sAlertType == "service_provider")
	{
		return jRawDetails.category + ": " + jRawDetails.message;
	}	
	else if (sAlertType == "public_disclosure")
	{
		return jRawDetails.category + ": " + jRawDetails.subcategory + " (" + jRawDetails.discovery_date + ") " + jRawDetails.message;
	}
	else
	{
		return "unknown type";
	}
}

function GetDirectionNumber(start,end)
{
	if (typeof start == 'undefined' || typeof end == 'undefined'){
		return "?";
	}
	var istart = parseInt(start);
	var iend = parseInt(end);

	if (istart > iend){ //Down
		return "<font color=red><b>&darr;</b></font>";  //Down arrow
	}
	else if (istart < iend){ //Up
		return "<font color=green><b>&uarr;</b></font>"; //Up arrow
	}
	else{
		return "?";
	}
}


function GetDirectionLetter(start,end)
{
	if (typeof start == 'undefined' || typeof end == 'undefined'){
		return "?";
	}

	//Using ascii converion to test direction
		if (start.charCodeAt(0) > end.charCodeAt(0)) //B --> A
	{
		return "<font color=green><b>&uarr;</b></font>"; //Up arrow
	}
	else if (start.charCodeAt(0) < end.charCodeAt(0)) //C --> D
	{
		return "<font color=red><b>&darr;</b></font>"; //Down arrow
	}
	else
	{
		return "?";
	}
}



function DetermineExportPath(callBack)
{
    if (params['bIsSaaS'] == "true")
    {
        //This functionality is only for SaaS/Hosted clients running this script outside of Archer (Node.js Prompt).
        if (aBitSightAlerts.length == 0)
        {
            LogInfo('Nothing to send back to Archer...exiting.');
            LogSaaSSuccess();
        }
        else
        {
            getArcherAlertsFieldIDs(callBack);
        }
    }
    else
    {
        //Return data back to Archer data feed
        LogInfo('Non-SaaS - returning data to Archer.');	
        ReturnDataToArcher(callBack);
    }
}



/********************************************************/
/********	ReturnDataToArcher
/********************************************************/
var ReturnDataToArcher = function(callBack)
{
	LogInfo('--------------Non-SaaS - ReturnDataToArcher-------------------------------');
	LogInfo('--------------Generating XML for Archer Records---------------------------');
	var start = Buffer.from('<Records>\n', 'utf8');
	var xmlData = jsonArrayToXMLBuffer(aBitSightAlerts,'Record');
	var end = Buffer.from('</Records>\n', 'utf8');
	var returntoArcher = Buffer.concat([start,xmlData,end]);

	var sXMLReturn = returntoArcher.toString('utf8');
	//LogInfo("XML: " + sXMLReturn);
	
    LogInfo("Finish with callback...");
	callBack(false,sXMLReturn);
}



/********************************************************/
/********	getArcherAlertsFieldIDs
/********************************************************/
var getArcherAlertsFieldIDs = function(callBack)
{
	LogInfo('----------------------------getArcherAlertsFieldIDs----------------------------');
	var getAlertsModuleID = function(callBack)
	{
		LogInfo('----------------------------getAlertsModuleID----------------------------');
		var url =  params['archer_webroot'] + params['archer_rest_root'] + params['archer_applicationpath'];

		var headers = {	'Authorization' : 'Archer session-id="' + sSessionToken + '"',
						'Content-type': 'application/json',
						'Accept': 'application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8'};
		
		var odataquery = "?$select=Name,Id,Guid&$filter=Name eq '" + params['archer_BitSightAlertsApp'] + "'";
		var postBody = {
			"Value": odataquery
		};

		LogInfo('URL5: ' + url);
		LogVerbose('BODY5: ' + jsonToString(postBody));
		//LogVerbose('HEAD: ' + headers);

		webCall(url,'GET',headers,jsonToString(postBody),function(err,h,b)
		{
			if(err)
			{
				LogError("ERR5: " + err);
				//LogError("HEAD5: " + h);
				LogError("BODY5: " + b);
			}
			else
			{
				LogVerbose("Raw Report Results: " + b);
				var resultJSON = JSON.parse(b);

				if(typeof resultJSON != 'undefined' && typeof resultJSON[0].RequestedObject != 'undefined' && resultJSON[0].RequestedObject.Id != 'undefined')
				{
					var iModuleID = resultJSON[0].RequestedObject.Id; //Get the content ID
					LogInfo("iModuleID: " + iModuleID);
					getFieldIDs(iModuleID,callBack);
				}
				else
				{
					LogError("ERROR Obtaining BitSight Alerts module ID: " + b);
				}
			}
		});
	}

	var getFieldIDs = function(iModuleID,callBack)
	{
		LogInfo('----------------------------getAlertsFieldIDs----------------------------');
		var url =  params['archer_webroot'] + params['archer_rest_root'] + params['archer_fielddefinitionapppath'] + iModuleID;

		var headers = {	'Authorization' : 'Archer session-id="' + sSessionToken + '"',
						'Content-type': 'application/json',
						'Accept': 'application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8'};
		
		var postBody = {
			"Value": "?$orderby=Name"
		  };

		LogInfo('URL6: ' + url);
		LogVerbose('BODY6: ' + jsonToString(postBody));
		//LogVerbose('HEAD: ' + headers);

		webCall(url,'GET',headers,jsonToString(postBody),function(err,h,b)
		{
			if(err)
			{
				LogError("ERR6: " + err);
				//LogError("HEAD6: " + h);
				LogError("BODY6: " + b);
				callBack(true,null);
			}
			else
			{
				LogVerbose("Alerts App Definition: " + b);
				//LogInfo("Raw Report Results: " + b);
				var resultJSON = JSON.parse(b);

				if(typeof resultJSON != 'undefined' && typeof resultJSON[0].RequestedObject != 'undefined')
				{
					LogVerbose("Archer Returned good app data");

					//Thankfully LevelID is included in every field and since Alerts is not a leveled app, we can just grab it off the first field
					SaaSParams['AlertsLevelID'] = resultJSON[0].RequestedObject.LevelId;

					for(var iField in resultJSON)
					{
						var sfieldname = resultJSON[iField].RequestedObject.Name.toLowerCase();
						var salias = resultJSON[iField].RequestedObject.Alias.toLowerCase();
						var sguid = resultJSON[iField].RequestedObject.Guid.toLowerCase();

						//LogVerbose("*Looking for: " + sfieldname);

						//Check field name, alias, and guid since client might rename one or more items and GUID could theoretically be used in another app
						if (sfieldname == "alert date" || salias == "alert_date" || sguid == "507ff4fc-da20-4eae-9845-0ceaff68ba4b")
						{
							//LogVerbose("Got Alert Date");
							SaaSParams['alert_date'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "alert id" || salias == "alert_id" || sguid == "47758960-7859-4d35-af8d-d9d17c2d60db")
						{
							SaaSParams['alert_id'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "alert trigger" || salias == "alert_trigger" || sguid == "61cc2865-7482-4e9a-be75-05b4bafa960a")
						{
							SaaSParams['alert_trigger'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "alert type" || salias == "alert_type" || sguid == "5a535a5a-0d66-4d0a-b7c6-59be380db47b")
						{
							SaaSParams['alert_type'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "category name" || salias == "category_name" || sguid == "ee67a048-b1a2-4e4c-8593-4866fb0a6979")
						{
							SaaSParams['category_name'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "details" || salias == "details" || sguid == "c5ad993a-933d-4cf8-abc4-669d64931127")
						{
							SaaSParams['details'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "disclosure severity" || salias == "disclosure_severity" || sguid == "b141bc49-207d-4ba3-956c-16394b57eafe")
						{
							SaaSParams['disclosure_severity'] = resultJSON[iField].RequestedObject.Id;
							ArcherValuesLists[ArcherValuesLists.length]={
								'FieldName':'disclosure_severity',
								'ValuesListID': resultJSON[iField].RequestedObject.RelatedValuesListId,
								'Prefix':'disclosure_severity_'
							};
						}
						else if (sfieldname == "disclosure/publication date" || salias == "disclosurepublication_date" || sguid == "d0b85e9a-da8d-4a1a-9670-3cc11061862")
						{
							SaaSParams['disclosure_publication_date'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "event type" || salias == "event_type" || sguid == "af3eb42c-1ce2-4839-a9a9-0cdc8f166ed5")
						{
							SaaSParams['event_type'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "event type description" || salias == "event_type_description" || sguid == "a3c0202b-5356-4281-8f14-b518e59a7c4e")
						{
							SaaSParams['event_type_description'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "folder name" || salias == "folder_name" || sguid == "bddd15b5-640f-4318-9ca0-e6acd062228b")
						{
							SaaSParams['folder_name'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "rating change" || salias == "rating_change" || sguid == "d6b4315f-29ca-4314-9f78-4240cdb4fc4a")
						{
							SaaSParams['rating_change'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "raw" || salias == "raw" || sguid == "717096bb-3482-4914-a6cd-651b66c57167")
						{
							SaaSParams['raw'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "record type" || salias == "record_type" || sguid == "56e0baa7-4085-40c4-b8e2-befa0548e74f")
						{
							SaaSParams['record_type'] = resultJSON[iField].RequestedObject.Id;
							ArcherValuesLists[ArcherValuesLists.length]={
								'FieldName':'record_type',
								'ValuesListID': resultJSON[iField].RequestedObject.RelatedValuesListId,
								'Prefix':'record_type_'
							};							
						}
						else if (sfieldname == "related bitsight portfolio" || sfieldname == "company name" || salias == "related_bitsight_portfolio" || sguid == "9ef269f0-8bba-4efe-a8ff-7e038f78707b")
						{
							SaaSParams['related_bitsight_portfolio'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "severity" || salias == "severity" || sguid == "b975abae-de52-46ec-9f16-948548543dc0")
						{
							SaaSParams['severity'] = resultJSON[iField].RequestedObject.Id;
							ArcherValuesLists[ArcherValuesLists.length]={
								'FieldName':'severity',
								'ValuesListID': resultJSON[iField].RequestedObject.RelatedValuesListId,
								'Prefix':'severity_'
							};								
						}
						else if (sfieldname == "url" || salias == "url" || sguid == "f91085f0-63f6-492b-8cf9-ad2da7eeb5dc")
						{
							SaaSParams['url'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "uniqueid" || salias == "uniqueid" || sguid == "9880a760-6d79-4163-87f3-d813989378f7")
						{
							SaaSParams['uniqueid'] = resultJSON[iField].RequestedObject.Id;
						}						
					}

					if (bVerboseLogging === true)
					{
						var fs = require('fs'); 			
						fs.writeFileSync("logs\\" + appPrefix + "\\" + "aParameters1.json", JSON.stringify(SaaSParams));
						LogInfo("Saved to logs\\" + appPrefix + "\\" + "aParameters1.json file");
					}
			
					LogInfo("SaaSParams: " + jsonToString(SaaSParams));
					
					getValuesListValues(0,callBack);				
				}
				else
				{
					LogError("ERROR Obtaining Alerts field definition: " + b);
				}
			}
		});
	}

	var getValuesListValues = function(currentValuesList,callBack)
	{
		LogInfo('----------------------------getValuesListValues----------------------------');
		LogInfo('Getting VL: ' + ArcherValuesLists[currentValuesList].FieldName);
		var valueslistID = ArcherValuesLists[currentValuesList].ValuesListID;
		var url =  params['archer_webroot'] + params['archer_rest_root'] + params['archer_valueslistvaluepath'] + valueslistID;

		var headers = {	'Authorization' : 'Archer session-id="' + sSessionToken + '"',
						'Content-type': 'application/json',
						'Accept': 'application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8'};
		
		var postBody = {};

		LogInfo('URL7: ' + url);

		webCall(url,'GET',headers,jsonToString(postBody),function(err,h,b)
		{
			if(err)
			{
				LogError("ERR7: " + err);
				//LogError("HEAD7: " + h);
				LogError("BODY7: " + b);
				callBack(true,null);
			}
			else
			{
				LogVerbose("ValuesList Full Definition: " + b);
				//LogInfo("Raw Report Results: " + b);
				var resultJSON = JSON.parse(b);

				if(typeof resultJSON != 'undefined' && typeof resultJSON[0].RequestedObject != 'undefined')
				{
					LogVerbose("Archer Returned good values list data");

					var id = "";
					var name = "";
					var prefix = ArcherValuesLists[currentValuesList].Prefix;

					for(var i in resultJSON)
					{
						id = resultJSON[i].RequestedObject.Id;
						name = resultJSON[i].RequestedObject.Name;

						//Format name of value to remove spaces and /
						name = name.replace(/\//g, '');
						name = name.replace(/ /g, '');
						
						name = prefix + name; //Set the name of the parameter to fieldname_Value
						SaaSParams[name] = id;//Add to our parameters

					}

					//Iterate through if there were multiple
					//Did we hit the max?
					LogInfo("Current=" + currentValuesList + " Max=" + ArcherValuesLists.length);
					if(currentValuesList >= ArcherValuesLists.length-1)
					{
						LogInfo("Hit maxResults of " + ArcherValuesLists.length);				
						if (bVerboseLogging === true)
						{
							var fs = require('fs'); 			
							fs.writeFileSync("logs\\" + appPrefix + "\\" + "aParameters2.json", JSON.stringify(SaaSParams));
							LogInfo("Saved to logs\\" + appPrefix + "\\" + "aParameters2.json file");
						}

						formatDataForSaaSUpload(callBack);
					}
					else
					{
						//Still have more values lists to iterate through...
						currentValuesList++; //Increment before running again
						LogInfo("Iterating through next ValuesList=" + currentValuesList);				
						getValuesListValues(currentValuesList,callBack)
					}

				}
				else
				{
					LogError("ERROR Obtaining Portfolio field values list ids: " + b);
				}
			}
		});
	}

	//Start of all the functions to get ids
	getAlertsModuleID(callBack);
}





/********************************************************/
/********	formatDataForSaaSUpload
/********************************************************/
var formatDataForSaaSUpload = function(callBack)
{
	LogInfo('----------------------formatDataForSaaSUpload START------------------------------');

	var postBody = [];  //Array formatted as the postBody that will loop through to create records later

	//Retrieve the Archer Field ID's setup for the postbody
	var LevelID = SaaSParams['AlertsLevelID'];
    var alert_id = SaaSParams['alert_id'];
	var alert_date = SaaSParams['alert_date'];
    var alert_trigger = SaaSParams['alert_trigger'];
    var alert_type = SaaSParams['alert_type'];
	var alert_severity = SaaSParams['severity'];
	var folder_name = SaaSParams['folder_name'];
    var rating_change = SaaSParams['rating_change'];
    var related_bitsight_portfolio = SaaSParams['related_bitsight_portfolio'];
    var details = SaaSParams['details'];
	var raw = SaaSParams['raw'];
	var record_type = SaaSParams['record_type'];
	var record_type_alert = SaaSParams['record_type_Alert'];
	var uniqueid = SaaSParams['uniqueid'];

	LogInfo("Count of aBitSightAlerts: " + aBitSightAlerts.length);
	
	if (aBitSightAlerts.length == 0)
	{
		LogInfo('Nothing to send back to Archer...exiting.');
		LogSaaSSuccess();
	}
	else
	{
		for(var i in aBitSightAlerts)
		{
			LogInfo("  - Alert = " + i);
			var iArcherContentID = parseInt(aBitSightAlerts[i].ArcherContentID);

			postBody[i] = {
				"Content": {
					"LevelId": LevelID,
					"Tag": "BitSightAlerts",
					"FieldContents": {
						[related_bitsight_portfolio]:
						{
							"Type": ArcherInputTypes['crossref'],
							"Tag": "Related BitSight Portfolio - " + aBitSightAlerts[i].ArcherCompanyname, //Archer Content ID of the Related BitSight Portfolio record
							"Value": [
									{
										"ContentId":iArcherContentID
									}
								],
							"FieldID": related_bitsight_portfolio
						},
						[record_type]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "record_type",
							"Value": {
								"ValuesListIds": [
									record_type_alert
								]
							  },
							"FieldID": record_type
						},						
						[alert_id]:
						{
							"Type": ArcherInputTypes['text'],
							"Tag": "alert_id",
							"Value": aBitSightAlerts[i].guid,
							"FieldID": alert_id
						},
						[alert_type]:
						{
							"Type": ArcherInputTypes['text'],
							"Tag": "alert_type",
							"Value": aBitSightAlerts[i].alert_type,
							"FieldID": alert_type
						},
						[alert_severity]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "severity",
							"Value": {
								"ValuesListIds": [
									GetSeverityValuesListLookup(aBitSightAlerts[i].severity)
								]
							  },
							"FieldID": alert_severity
						},
						[rating_change]:
						{
							"Type": ArcherInputTypes['numeric'],
							"Tag": "rating_change",
							"Value": aBitSightAlerts[i].rating_change,
							"FieldID": rating_change
						},
						[folder_name]:
						{
							"Type": ArcherInputTypes['text'],
							"Tag": "folder_name",
							"Value": aBitSightAlerts[i].folder_name,
							"FieldID": folder_name
						},
						[alert_date]:
						{
							"Type": ArcherInputTypes['date'],
							"Tag": "alert_date",
							"Value": aBitSightAlerts[i].alert_date,
							"FieldID": alert_date
						},
						[alert_trigger]:
						{
							"Type": ArcherInputTypes['text'],
							"Tag": "alert_trigger",
							"Value": aBitSightAlerts[i].trigger,
							"FieldID": alert_trigger
						},
						[details]:
						{
							"Type": ArcherInputTypes['text'],
							"Tag": "details",
							"Value": aBitSightAlerts[i].details,
							"FieldID": details
						},
						[raw]:
						{
							"Type": ArcherInputTypes['text'],
							"Tag": "raw",
							"Value": aBitSightAlerts[i].raw,
							"FieldID": raw
						},
						[uniqueid]:
						{
							"Type": ArcherInputTypes['text'],
							"Tag": "uniqueid",
							"Value": aBitSightAlerts[i].guid,
							"FieldID": uniqueid
						}						
					}
				}
			}; //End of postBody

			//LogVerbose("postBody[" + i + "] = " +  jsonToString(postBody[i]));

		} //End of for loop for iteration

		return SaaSuploadToArcher(postBody,callBack);
	
	}//End of if statement when there is something to return to Archer

} //End of formatDataForSaaSUpload function

/********************************************************/
/********	SaaSuploadToArcher
/********************************************************/
var SaaSuploadToArcher = function(postBody,callBack)
{
	LogInfo("STARTING ON RECORD #: " + iCurrentAlert);

	var sMETHOD = "POST";
	LogInfo('----------------------------CREATE RECORD----------------------------');
	    
	var url =  params['archer_webroot'] + params['archer_rest_root'] + params['archer_contentpath'];

	var headers = {	'Authorization' : 'Archer session-id="' + sSessionToken + '"',
					'Content-type': 'application/json',
					'Accept': 'application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8'};
    
    LogInfo('URL8: ' + url);
	//LogVerbose('BODY8: ' + jsonToString(postBody[iCurrentAlert]));  //Since this is being saved as a file, no need to output to console.

	if (bVerboseLogging === true)
	{
		var fs = require('fs'); 			
		fs.writeFileSync("logs\\" + appPrefix + "\\" + "JSON_PostBody_" + iCurrentAlert + ".json", jsonToString(postBody[iCurrentAlert]));
		LogInfo("Saved to logs\\" + appPrefix + "\\" + "JSON_PostBody_" + iCurrentAlert + ".json file");
	}	

    webCall(url,sMETHOD,headers,jsonToString(postBody[iCurrentAlert]),function(err,h,b)
    {
        if(err)
        {
            LogError("ERR8: " + err);
            //LogError("HEAD8: " + h);
            LogError("BODY8: " + b);
        }
        else
        {
            //LogInfo("Raw Report Results: " + b);
            var resultJSON = JSON.parse(b);

            //Just for testing...save to file...
            if (bVerboseLogging === true)
            {
                var fs = require('fs'); 			
                fs.writeFileSync("logs\\" + appPrefix + "\\" + "ResultJSONCreateRecord" + iCurrentAlert + ".json", JSON.stringify(resultJSON));
                LogInfo("Saved to logs\\" + appPrefix + "\\" + "ResultJSONCreateRecord" + iCurrentAlert + ".json file");
            }
            
			var bIsSuccessful = false;
			if (typeof resultJSON == 'undefined' ||
				typeof resultJSON.IsSuccessful == 'undefined')
			{
				bIsSuccessful = false;
			}
			else if (resultJSON.IsSuccessful == true)
			{
				bIsSuccessful = true;
			}

			if (bIsSuccessful)
			{
				var iContentID = resultJSON.RequestedObject.Id; //Get the content ID
				LogInfo("iContentID created: " + iContentID);
			}
            else
			{
				//Note, this may not be a problem if the record already existed. Evaluate the message and determine next steps.
				
				var msg = "";
				if (typeof resultJSON.ValidationMessages == 'undefined' ||
					typeof resultJSON.ValidationMessages[0] == 'undefined' ||
					typeof resultJSON.ValidationMessages[0].ResourcedMessage == 'undefined')
				{
					msg = "No error message from Archer.";
				}else{
					msg = resultJSON.ValidationMessages[0].ResourcedMessage;
				}

				if (msg == "The value in the following field must be unique: UniqueID")
				{
					LogInfo("Record not created because it was a duplicate. Message: " + msg);
				}else{
					LogWarn("Record not created. Message: " + msg);
				}
			}

			//Iterate through if there were multiple records to create
			//Did we hit the max?		
			LogInfo("Current=" + iCurrentAlert + " Max=" + postBody.length);
			if(iCurrentAlert >= postBody.length-1)
			{
				LogInfo("Hit maxResults of " + postBody.length);
				LogSaaSSuccess();
                LogInfo("SUCCESS: SaaS IUR Records Created");
                return true;
			}
			else
			{
				//Still have more records to iterate through...
				LogInfo("Iterating through next record=" + iCurrentAlert);
				iCurrentAlert++; //Increment before running again
				return SaaSuploadToArcher(postBody);
			}
        }
    });
} //End of uploadToArcher function




/************************************************************************************************************************************************************************************************************************************/
/******** DATAFEED BEGINS HERE
/************************************************************************************************************************************************************************************************************************************/
var lrt = init();
ArcherAPI_Login(dfCallback);